
import os
import re
from pathlib import Path


from qgis.PyQt import uic, QtWidgets
from qgis.PyQt.QtWidgets import QApplication, QDialog, QMessageBox, QWidget


import numpy as np
import pandas as pd
import geopandas as gpd
from shapely.geometry.base import BaseGeometry


import common
from urbanq.logging.logging_config import logger
from urbanq.function.qss import gradient_style, default_style


from urbanq.function.file import (
    export_gdf,
    keep_columns_gdf,
    load_geojson_gdf,
    load_txt_or_csv_df,
    load_json_df_or_gdf,
    load_layer_or_shp_gdf,
    update_shapefile_layer,
    df_to_empty_geometry_gdf,
)

from urbanq.function.widgetutils import (
    show_progress,
    update_progress,
)

from urbanq.function.geo import (
    normalize_null_values,
)


from urbanq.menu.autoUI.fileRread_dockwidget import fileRreadDockWidget
from urbanq.menu.autoUI.fileSave_dockwidget import fileSaveDockWidget
from urbanq.menu.autoUI.fileSetting_dockwidget import fileSettingDockWidget
from urbanq.menu.autoUI.ImageDescription_dockwidget import ImageDescriptionDockWidget



FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'DataAggregationAndFiltering_dockwidget_base.ui'))


class DataAggregationAndFilteringDockWidget(QDialog, FORM_CLASS):  
    def __init__(self, parent=None):
        
        super(DataAggregationAndFilteringDockWidget, self).__init__(parent)  
        
        
        
        
        
        self.setupUi(self)

        
        show_progress(self.progressBar, False)

        
        self.menuPushButton.setProperty("class", "boldText")
        self.nextStepPushButton.setProperty("class", "boldText")
        self.previousStepPushButton.setProperty("class", "boldText")

        
        self.menuPushButton.clicked.connect(self.go_back_to_data_conversion)

        
        self.nextStepPushButton.clicked.connect(lambda: self.next_previous_clicked(1))
        self.nextStepPushButton.clicked.connect(lambda: self.update_current_progress(self.stackedWidget.currentIndex()))
        self.nextStepPushButton.clicked.connect(lambda: self.load_menu_ui(self.stackedWidget.currentIndex()))

        
        self.previousStepPushButton.clicked.connect(lambda: self.next_previous_clicked(-1))
        self.previousStepPushButton.clicked.connect(lambda: self.update_current_progress(self.stackedWidget.currentIndex()))
        self.previousStepPushButton.clicked.connect(lambda: self.load_menu_ui(self.stackedWidget.currentIndex()))

        
        self.job_index = common.job_info.get("job_index") if common.job_info else None
        self.job_title = common.job_info.get("job_title") if common.job_info else None

        
        self.option = self.get_widget_option(self.job_index, self.job_title)

        
        self.pages_and_files = self.configure_pages_and_files()

        
        self.update_current_progress(0)

        
        self.stackedWidget.setCurrentIndex(0)

        
        self.load_menu_ui(0)

    
    
    

    def configure_pages_and_files(self):
        
        try:
            pages = []

            
            pages.append((True, self.current_step_1, ImageDescriptionDockWidget, None, None))

            
            pages.append((True, self.current_step_2, fileRreadDockWidget, self.option, None))

            
            read_required = any([
                self.option["setting_by_text"],
                self.option["setting_by_array"],
                self.option["setting_by_expression"],
                self.option["setting_by_section"]["enabled"],
                self.option["setting_by_numeric"]["enabled"],
                self.option["setting_by_combo"]["enabled"],
            ])
            pages.append((read_required, self.current_step_3, fileSettingDockWidget, self.option, None))

            
            save_required = any([
                self.option["output_by_file"],
                self.option["output_by_field"],
                self.option["output_by_table"]
            ])
            pages.append((save_required, self.current_step_4, fileSaveDockWidget, self.option, None))

            return pages

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def go_back_to_data_conversion(self):
        
        try:
            from urbanq.menu.dataConversion.dataConversion_dockwidget import dataConversionDockWidget  
            parent_ui = dataConversionDockWidget(self)  
            main_page_layout = self.parent().parent().findChild(QWidget, "page_dataConversion").layout()
            if main_page_layout:
                
                for i in reversed(range(main_page_layout.count())):
                    main_page_layout.itemAt(i).widget().deleteLater()
                main_page_layout.addWidget(parent_ui)

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def load_menu_ui(self, index):
        
        try:
            widget_enabled, widget_process, widget_class, widget_option, widget_instance = self.pages_and_files[index]
            page = self.stackedWidget.widget(index)

            
            if widget_instance is None:

                
                widget_instance = widget_class(self, self.option)
                page.layout().addWidget(widget_instance)
                self.pages_and_files[index] = (
                    self.pages_and_files[index][0],
                    self.pages_and_files[index][1],
                    self.pages_and_files[index][2],
                    self.pages_and_files[index][3],
                    widget_instance
                )

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def update_current_progress(self, index):
        
        try:
            step = 1
            for i, (widget_enabled, widget_process, _, _, _) in enumerate(self.pages_and_files):
                if not widget_enabled:
                    widget_process.hide()
                    continue
                else:
                    updated_text = re.sub(r"\[\d+단계\]", f"[{step}단계]", widget_process.text())
                    widget_process.setText(updated_text)
                    step += 1

                
                widget_process.show()

                if i == index:
                    widget_process.setStyleSheet(gradient_style)
                else:
                    widget_process.setStyleSheet(default_style)

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def get_safe_page_index(self, current_index: int, direction: int) -> int:
        
        try:
            new_index = current_index

            while True:
                
                new_index += direction

                
                new_index = max(0, min(new_index, len(self.pages_and_files) - 1))

                
                if self.pages_and_files[new_index][0]:
                    return new_index

                
                if new_index == 0 and direction == -1:
                    return current_index

                
                if new_index == len(self.pages_and_files) - 1 and direction == 1:
                    return current_index

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def next_previous_clicked(self, direction):
        
        def get_last_valid_page_index(pages_and_files) -> int:
            
            for i in reversed(range(len(pages_and_files))):
                if pages_and_files[i][0]:
                    return i
            return -1  

        try:
            
            current_index = self.stackedWidget.currentIndex()

            
            if self.pages_and_files[current_index][0]:
                instance = self.pages_and_files[current_index][4]
                if direction > 0 and not instance.set_fileResults():
                    return

            
            new_index = self.get_safe_page_index(current_index, direction)

            
            last_page_index = get_last_valid_page_index(self.pages_and_files)

            
            self.nextStepPushButton.setText("실행하기 " if new_index == last_page_index else "다음 단계 ▶")

            
            self.stackedWidget.setCurrentIndex(new_index)

            
            if current_index == last_page_index and direction > 0:
                self.run_job_process()

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    
    
    

    def get_file_data_frame(self, source_file_type, source_file_path, file_path, file_encoding, file_delimiter, file_has_header):
        
        try:
            
            gdf = None

            
            if source_file_type == "shp":
                gdf = load_layer_or_shp_gdf(shp_path=file_path, file_encoding=file_encoding)

            
            elif source_file_type == "layer":
                qgs_project_layer = source_file_path
                gdf = load_layer_or_shp_gdf(layer=qgs_project_layer, file_encoding=file_encoding)

            
            elif source_file_type == "json":
                df, _ = load_json_df_or_gdf(file_path=file_path, file_encoding=file_encoding)
                gdf = df_to_empty_geometry_gdf(df)

            
            elif source_file_type == "geojson":
                gdf = load_geojson_gdf(file_path=file_path, file_encoding=file_encoding)

            
            elif source_file_type == "txt":
                df = load_txt_or_csv_df(file_path, file_encoding, file_delimiter, file_has_header)
                gdf = df_to_empty_geometry_gdf(df)

            
            elif source_file_type == "csv":
                df = load_txt_or_csv_df(file_path, file_encoding, file_delimiter, file_has_header)
                gdf = df_to_empty_geometry_gdf(df)

            
            elif source_file_type == "folder":
                df = load_txt_or_csv_df(file_path, file_encoding, file_delimiter, file_has_header)
                gdf = df_to_empty_geometry_gdf(df)

            if gdf is None:
                return

            return gdf

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def run_job_process(self):
        
        try:
            
            show_progress(self.progressBar)

            
            total_files = len(common.fileInfo_1.file_preview)  
            steps_per_file = 4  
            total_steps = total_files * steps_per_file  
            base_progress = 20  
            step_weight = (100 - base_progress) / total_steps  
            current_step = 0  

            
            source_file_type, source_file_path, _ = common.fileInfo_1.file_record.get_record()
            result_file_type, result_file_path, _ = common.fileInfo_1.result_record.get_record()

            
            status_flags = []  
            for index, file_preview in enumerate(common.fileInfo_1.file_preview):

                
                file_path, file_encoding, file_delimiter, file_has_header = file_preview.get_info()
                current_step += 1
                update_progress(self.progressBar, int(base_progress + current_step * step_weight))

                
                if source_file_type == "folder":
                    
                    file_name_with_ext = os.path.basename(file_path)
                    new_file_path = os.path.join(result_file_path, file_name_with_ext)
                elif result_file_type == "layer":
                    new_file_path = file_path
                else:
                    new_file_path = result_file_path

                
                gdf = self.get_file_data_frame(source_file_type, source_file_path, file_path, file_encoding, file_delimiter, file_has_header)
                current_step += 1
                update_progress(self.progressBar, int(base_progress + current_step * step_weight))

                
                result = self.run_job_by_index(gdf, index)
                current_step += 1
                update_progress(self.progressBar, int(base_progress + current_step * step_weight))

                
                if result is None:
                    status_flags.append(False)
                    break
                elif result is True:
                    
                    
                    status_flags.append(True)

                try:
                    
                    if result_file_type == 'layer':

                        
                        layer_widget = self.pages_and_files[1][4].get_qgs_layer_widget()

                        
                        layer_widget_index = layer_widget.currentIndex()

                        
                        layer = source_file_path

                        
                        new_layer = update_shapefile_layer(layer, result)

                        
                        if 0 <= layer_widget_index < layer_widget.count():
                            layer_widget.setCurrentIndex(layer_widget_index)

                        
                        common.fileInfo_1.file_record.file_path[result_file_type] = new_layer

                        
                        status_flags.append(True)

                    else:
                        
                        if new_file_path:

                            
                            if isinstance(result, gpd.GeoDataFrame):
                                export_success = export_gdf(result, new_file_path)

                                
                                status_flags.append(export_success)

                            elif isinstance(result, list) and result:
                                
                                file_type, _, file_name = common.fileInfo_1.file_record.get_record()
                                base_dir = Path(new_file_path)
                                base_name = Path(file_name).stem
                                ext = f".{file_type}"

                                
                                export_success = []
                                for i, part in enumerate(result, start=1):
                                    output_path = base_dir / f"{base_name}_{i:03d}{ext}"
                                    export_success.append(export_gdf(part, output_path))

                                
                                status_flags.append(all(export_success))

                            else:
                                
                                QMessageBox.information(self, "파일 오류", "파일 저장 중 오류가 발생했습니다.", QMessageBox.Ok)
                                status_flags.append(False)

                except Exception as e:
                    
                    QMessageBox.information(self, "파일 오류", f"GeoDataFrame export 실패: {e}", QMessageBox.Ok)
                    status_flags.append(False)

                
                current_step += 1
                update_progress(self.progressBar, int(base_progress + current_step * step_weight))

            
            if status_flags and all(status_flags):
                update_progress(self.progressBar, 100)  
                QMessageBox.information(self, "알림", "축하합니다. 작업이 완료했습니다!", QMessageBox.Ok)

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

        finally:
            show_progress(self.progressBar, False)

    
    
    

    
    @staticmethod
    def convert_file_type(gdf):
        return gdf

    
    def get_top_n_rows(self, gdf, n):
        
        try:
            
            gdf_copy = gdf.copy()

            
            gdf_copy = normalize_null_values(gdf_copy)

            
            n = int(float(n))

            
            if n <= 0:
                QMessageBox.information(self, "작업 오류", "레코드 개수는 1 이상으로 입력해 주세요.", QMessageBox.Ok)
                return None

            
            if isinstance(n, int) and n > 0:
                if n >= len(gdf_copy):
                    return gdf_copy  
                else:
                    return gdf_copy.head(n).copy()

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "파일을 작업 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    
    def get_top_or_bottom_n_rows_by_field(self, gdf, field_name, n, is_sort=True):
        
        try:
            
            try:
                n = int(float(n))
                
                if n <= 0:
                    QMessageBox.information(self, "작업 오류", "레코드 개수는 1 이상으로 입력해 주세요.", QMessageBox.Ok)
                    return None

            except Exception as e:
                QMessageBox.information(self, "작업 오류", "레코드 개수는 숫자 형식으로 입력해 주세요.", QMessageBox.Ok)
                logger.error("에러 발생: %s", e, exc_info=True)
                return None

            
            gdf_copy = gdf.copy()

            
            gdf_copy = normalize_null_values(gdf_copy)

            
            try:
                
                
                
                
                gdf_copy[field_name] = pd.to_numeric(gdf_copy[field_name], errors="coerce").fillna(0).astype(float)

            except Exception as e:
                QMessageBox.information(self, "작업 오류", "해당 필드를 숫자 형식으로 변환할 수 없습니다.\n문자 또는 특수문자가 포함되어 있는지 확인해 주세요.", QMessageBox.Ok)
                logger.error("에러 발생: %s", e, exc_info=True)
                return None

            
            series = gdf_copy[field_name]
            if pd.api.types.is_float_dtype(series) and (series.dropna() % 1 == 0).all():
                gdf_copy[field_name] = series.astype("Int64")
            else:
                gdf_copy[field_name] = series

            
            if is_sort:
                
                
                
                return gdf_copy.sort_values(by=field_name, ascending=False).head(n).copy()
            else:
                
                
                
                return gdf_copy.sort_values(by=field_name, ascending=True).head(n).sort_values(by=field_name, ascending=True).copy()

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "파일을 작업 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    
    def get_rows_with_keyword_in_field(self, gdf, field_name, keyword):
        
        try:
            
            gdf_copy = gdf.copy()

            
            gdf_copy = normalize_null_values(gdf_copy)

            
            try:
                gdf_copy[field_name] = gdf_copy[field_name].astype(str)

            except Exception as e:
                QMessageBox.information(self, "작업 오류", "해당 필드를 문자 형식으로 변환할 수 없습니다.", QMessageBox.Ok)
                logger.error("에러 발생: %s", e, exc_info=True)
                return None

            
            filtered_gdf = gdf_copy[gdf_copy[field_name].str.contains(keyword, na=False)]

            
            if filtered_gdf.empty:
                QMessageBox.information(self, "검색 결과", "해당 키워드를 포함하는 레코드가 없습니다.", QMessageBox.Ok)
                return None

            return filtered_gdf.copy()

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "파일을 작업 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    
    def extract_rows_by_range(self, gdf, start_no, end_no):
        
        try:
            
            try:
                s = int(start_no)
                e = int(end_no)
            except Exception:
                QMessageBox.information(self, "입력 오류", "시작 행과 끝 행은 정수로 입력해 주세요.\n예: 시작 1, 끝 100", QMessageBox.Ok)
                return None

            if s <= 0 or e <= 0:
                QMessageBox.information(self, "입력 오류", "행 번호는 1 이상의 정수여야 합니다.", QMessageBox.Ok)
                return None

            if s > e:
                QMessageBox.information(self, "입력 오류", "시작 행은 끝 행보다 클 수 없습니다.", QMessageBox.Ok)
                return None

            total_rows = len(gdf)
            if s > total_rows:
                QMessageBox.information(self, "입력 오류", f"시작 행({s})이 전체 행 수({total_rows})를 초과합니다.", QMessageBox.Ok)
                return None

            
            if e > total_rows:
                e = total_rows

            
            gdf_copy = gdf.copy()

            
            gdf_copy = normalize_null_values(gdf_copy)

            
            
            start_idx = s - 1
            end_idx_exclusive = e

            gdf_part = gdf_copy.iloc[start_idx:end_idx_exclusive].copy()

            if gdf_part.empty:
                QMessageBox.information(self, "처리 결과", "지정한 범위에 해당하는 데이터가 없습니다.", QMessageBox.Ok)
                return None

            return gdf_part

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "행 구간 추출 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("행 구간 추출 에러: %s", e, exc_info=True)
            return None

    
    def split_gdf_by_row_count(self, gdf, n):
        
        try:
            
            try:
                chunk_size = int(float(n))
            except Exception:
                QMessageBox.information(self, "입력 오류", "행 개수(n)는 정수로 입력해 주세요.\n예: 1000", QMessageBox.Ok)
                return None

            if chunk_size <= 0:
                QMessageBox.information(self, "입력 오류", "행 개수(n)는 1 이상의 정수여야 합니다.", QMessageBox.Ok)
                return None

            if gdf is None or getattr(gdf, "empty", True):
                QMessageBox.information(self, "입력 오류", "입력 데이터가 비어 있습니다.", QMessageBox.Ok)
                return None

            total_rows = len(gdf)
            if total_rows == 0:
                QMessageBox.information(self, "입력 오류", "입력 데이터가 비어 있습니다.", QMessageBox.Ok)
                return None

            
            gdf_copy = gdf.copy()

            
            gdf_copy = normalize_null_values(gdf_copy)

            
            parts = []
            start_idx = 0

            while start_idx < total_rows:
                end_idx = min(start_idx + chunk_size, total_rows)
                part = gdf_copy.iloc[start_idx:end_idx].copy()

                if not part.empty:
                    parts.append(part)

                start_idx = end_idx

            if not parts:
                QMessageBox.information(self, "처리 결과", "분할 결과가 없습니다.", QMessageBox.Ok)
                return None

            return parts

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "행 개수 기준 분할 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("행 개수 기준 분할 에러: %s", e, exc_info=True)
            return None

    
    
    

    @staticmethod
    def get_widget_option(job_index, job_title):
        
        try:
            option = None  
            job_title = job_title[2:]

            if job_index == 0:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": False,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": False,
                    "show_field_in_file": True,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,
                }
            if job_index == 1:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": True,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": False,
                    "show_field_in_file": True,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": True, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,

                    "SETTING_NUMERIC": [
                        '가져올 레코드 개수 설정',
                        '레코드 개수 입력: ',
                        '파일의 앞부분에서 불러올 레코드 개수를 입력해 주세요.'
                    ],
                }
            if job_index == 2:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": True,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": True,
                    "show_field_in_file": True,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": True, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,

                    "FILE_TUID": [
                        '상위 레코드 산정 기준 필드 선택',
                        '필드 선택: ',
                        '선택한 필드를 기준으로 상위 N개의 레코드가 산정됩니다.'
                    ],

                    "SETTING_NUMERIC": [
                        '가져올 상위 레코드 개수 설정',
                        '레코드 개수 입력: ',
                        '추출할 상위 레코드의 개수(N)를 입력해 주세요.'
                    ],
                }

            if job_index == 3:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": True,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": True,
                    "show_field_in_file": True,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": True, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,

                    "FILE_TUID": [
                        '하위 레코드 산정 기준 필드 선택',
                        '필드 선택: ',
                        '선택한 필드를 기준으로 하위 N개의 레코드가 산정됩니다.'
                    ],

                    "SETTING_NUMERIC": [
                        '가져올 하위 레코드 개수 설정',
                        '레코드 개수 입력: ',
                        '추출할 하위 레코드의 개수(N)를 입력해 주세요.'
                    ]
                }
            if job_index == 4:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": True,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": True,
                    "show_field_in_file": True,

                    "setting_by_text": True,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,

                    "FILE_TUID": [
                        '문자 또는 키워드 포함 필드 선택',
                        '필드 선택: ',
                        '문자 또는 키워드가 포함된 필드를 선택해 주세요.'
                    ],

                    "SETTING_TEXT": [
                        '포함할 문자 또는 키워드 입력',
                        '문자 또는 키워드 입력: ',
                        '필드 값에 포함된 특정 문자 또는 키워드를 입력해 주세요.'
                    ],
                }
            if job_index == 5:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": False,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": False,
                    "show_field_in_file": False,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": True, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,

                    "SETTING_SECTION": [
                        '행 범위 설정',
                        '시작 행: ',
                        '끝 행: ',
                        '지정한 시작 행(n)부터 끝 행(n+m)까지의 데이터만 선택하여 파일로 저장합니다.'
                    ],
                }
            if job_index == 6:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": False,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": False,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": False,
                    "show_field_in_file": False,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": True, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,

                    "file_to_fold": True,

                    "SETTING_NUMERIC": [
                        "파일 분할 기준 행 개수 설정",
                        "행 개수 입력:",
                        "입력한 행 개수(n)를 기준으로 파일이 여러 개로 분할됩니다."
                    ],
                }
            return option

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def run_job_by_index(self, gdf, file_preview_index):
        
        try:
            
            file_info = common.fileInfo_1

            
            setting_text = file_info.file_setting.get_text()
            setting_numeric = file_info.file_setting.get_numeric()
            setting_section_min, setting_section_max = file_info.file_setting.get_section()
            setting_combo = file_info.file_setting.get_combo()
            setting_array_string, setting_array_integer, setting_array_float = file_info.file_setting.get_array()

            
            source_file_type, source_file_path, source_file_name = file_info.file_record.get_record()

            
            file_preview = file_info.file_preview[file_preview_index]
            file_field_selection = file_preview.get_selection_field()
            file_uid = file_preview.get_file_uid()
            file_tuid = file_preview.get_file_tuid()
            file_is_field_check = file_preview.get_field_check()
            result_field = file_info.result_field

            
            gdf.columns = gdf.columns.astype(str)

            
            result = None
            if self.job_index == 0:
                gdf = keep_columns_gdf(gdf, file_field_selection) if file_is_field_check else gdf
                result = self.convert_file_type(gdf)

            elif self.job_index == 1:
                gdf = keep_columns_gdf(gdf, file_field_selection) if file_is_field_check else gdf
                result = self.get_top_n_rows(gdf, setting_numeric)

            elif self.job_index == 2:
                gdf = keep_columns_gdf(gdf, file_field_selection + [file_tuid]) if file_is_field_check else gdf
                result = self.get_top_or_bottom_n_rows_by_field(gdf, file_tuid, setting_numeric)

            elif self.job_index == 3:
                gdf = keep_columns_gdf(gdf, file_field_selection + [file_tuid]) if file_is_field_check else gdf
                result = self.get_top_or_bottom_n_rows_by_field(gdf, file_tuid, setting_numeric, False)

            elif self.job_index == 4:
                gdf = keep_columns_gdf(gdf, file_field_selection + [file_tuid]) if file_is_field_check else gdf
                result = self.get_rows_with_keyword_in_field(gdf, file_tuid, setting_text)

            elif self.job_index == 5:
                gdf = keep_columns_gdf(gdf, file_field_selection + [file_tuid]) if file_is_field_check else gdf
                result = self.extract_rows_by_range(gdf, setting_section_min, setting_section_max)

            elif self.job_index == 6:
                gdf = keep_columns_gdf(gdf, file_field_selection + [file_tuid]) if file_is_field_check else gdf
                result = self.split_gdf_by_row_count(gdf, setting_numeric)

            
            if result is None or result is False:
                return None

            return result

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)
            return None



