
import os
import re
from pathlib import Path
from typing import Optional


from qgis.PyQt import uic, QtWidgets
from qgis.PyQt.QtWidgets import QApplication, QDialog, QMessageBox, QWidget


from qgis.core import QgsVectorLayer


import numpy as np
import pandas as pd
import geopandas as gpd
from shapely.geometry.base import BaseGeometry


import common
from urbanq.logging.logging_config import logger
from urbanq.function.qss import gradient_style, default_style


from urbanq.function.file import (
    export_gdf,
    keep_columns_gdf,
    load_geojson_gdf,
    load_txt_or_csv_df,
    load_json_df_or_gdf,
    load_layer_or_shp_gdf,
    update_shapefile_layer,
    df_to_empty_geometry_gdf,
)

from urbanq.function.geo import (
    is_real_null,
    is_empty_value,
    normalize_null_values,
)

from urbanq.function.widgetutils import (
    show_progress,
    update_progress,
)


from urbanq.menu.autoUI.fileRread_dockwidget import fileRreadDockWidget
from urbanq.menu.autoUI.fileSave_dockwidget import fileSaveDockWidget
from urbanq.menu.autoUI.fileSetting_dockwidget import fileSettingDockWidget
from urbanq.menu.autoUI.ImageDescription_dockwidget import ImageDescriptionDockWidget
from urbanq.menu.spatialAnalysis.AddressConversion.address_conversion_client import batch_geocode


FORM_CLASS, _ = uic.loadUiType(os.path.join(
    os.path.dirname(__file__), 'AddressConversion_dockwidget_base.ui'))


class AddressConversionDockWidget(QDialog, FORM_CLASS):  
    def __init__(self, parent=None):
        
        super(AddressConversionDockWidget, self).__init__(parent)  
        
        
        
        
        
        self.setupUi(self)

        
        show_progress(self.progressBar, False)
        self.base_progress = 10

        
        self.menuPushButton.setProperty("class", "boldText")
        self.nextStepPushButton.setProperty("class", "boldText")
        self.previousStepPushButton.setProperty("class", "boldText")

        
        self.menuPushButton.clicked.connect(self.go_back_to_data_conversion)

        
        self.nextStepPushButton.clicked.connect(lambda: self.next_previous_clicked(1))
        self.nextStepPushButton.clicked.connect(lambda: self.update_current_progress(self.stackedWidget.currentIndex()))
        self.nextStepPushButton.clicked.connect(lambda: self.load_menu_ui(self.stackedWidget.currentIndex()))

        
        self.previousStepPushButton.clicked.connect(lambda: self.next_previous_clicked(-1))
        self.previousStepPushButton.clicked.connect(lambda: self.update_current_progress(self.stackedWidget.currentIndex()))
        self.previousStepPushButton.clicked.connect(lambda: self.load_menu_ui(self.stackedWidget.currentIndex()))

        
        self.job_index = common.job_info.get("job_index") if common.job_info else None
        self.job_title = common.job_info.get("job_title") if common.job_info else None

        
        self.option = self.get_widget_option(self.job_index, self.job_title)

        
        self.pages_and_files = self.configure_pages_and_files()

        
        self.update_current_progress(0)

        
        self.stackedWidget.setCurrentIndex(0)

        
        self.load_menu_ui(0)

    
    
    

    def configure_pages_and_files(self):
        
        try:
            pages = []

            
            pages.append((True, self.current_step_1, ImageDescriptionDockWidget, None, None))

            
            pages.append((True, self.current_step_2, fileRreadDockWidget, self.option, None))

            
            read_required = any([
                self.option["setting_by_text"],
                self.option["setting_by_array"],
                self.option["setting_by_expression"],
                self.option["setting_by_section"]["enabled"],
                self.option["setting_by_numeric"]["enabled"],
                self.option["setting_by_combo"]["enabled"],
            ])
            pages.append((read_required, self.current_step_3, fileSettingDockWidget, self.option, None))

            
            save_required = any([
                self.option["output_by_file"],
                self.option["output_by_field"],
                self.option["output_by_table"]
            ])
            pages.append((save_required, self.current_step_4, fileSaveDockWidget, self.option, None))

            return pages

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def go_back_to_data_conversion(self):
        
        try:
            from urbanq.menu.spatialAnalysis.spatialAnalysis_dockwidget import spatialAnalysisDockWidget  
            parent_ui = spatialAnalysisDockWidget(self)  
            main_page_layout = self.parent().parent().findChild(QWidget, "page_spatialAnalysis").layout()
            if main_page_layout:
                
                for i in reversed(range(main_page_layout.count())):
                    main_page_layout.itemAt(i).widget().deleteLater()
                main_page_layout.addWidget(parent_ui)

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def load_menu_ui(self, index):
        
        try:
            widget_enabled, widget_process, widget_class, widget_option, widget_instance = self.pages_and_files[index]
            page = self.stackedWidget.widget(index)

            
            if widget_instance is None:

                
                widget_instance = widget_class(self, self.option)
                page.layout().addWidget(widget_instance)
                self.pages_and_files[index] = (
                    self.pages_and_files[index][0],
                    self.pages_and_files[index][1],
                    self.pages_and_files[index][2],
                    self.pages_and_files[index][3],
                    widget_instance
                )

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def update_current_progress(self, index):
        
        try:
            step = 1
            for i, (widget_enabled, widget_process, _, _, _) in enumerate(self.pages_and_files):
                if not widget_enabled:
                    widget_process.hide()
                    continue
                else:
                    updated_text = re.sub(r"\[\d+단계\]", f"[{step}단계]", widget_process.text())
                    widget_process.setText(updated_text)
                    step += 1

                
                widget_process.show()

                if i == index:
                    widget_process.setStyleSheet(gradient_style)
                else:
                    widget_process.setStyleSheet(default_style)

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def get_safe_page_index(self, current_index: int, direction: int) -> int:
        
        try:
            new_index = current_index

            while True:
                
                new_index += direction

                
                new_index = max(0, min(new_index, len(self.pages_and_files) - 1))

                
                if self.pages_and_files[new_index][0]:
                    return new_index

                
                if new_index == 0 and direction == -1:
                    return current_index

                
                if new_index == len(self.pages_and_files) - 1 and direction == 1:
                    return current_index

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def next_previous_clicked(self, direction):
        
        def get_last_valid_page_index(pages_and_files) -> int:
            
            for i in reversed(range(len(pages_and_files))):
                if pages_and_files[i][0]:
                    return i
            return -1  

        try:
            
            current_index = self.stackedWidget.currentIndex()

            
            if self.pages_and_files[current_index][0]:
                instance = self.pages_and_files[current_index][4]
                if direction > 0 and not instance.set_fileResults():
                    return

            
            new_index = self.get_safe_page_index(current_index, direction)

            
            last_page_index = get_last_valid_page_index(self.pages_and_files)

            
            self.nextStepPushButton.setText("실행하기 " if new_index == last_page_index else "다음 단계 ▶")

            
            self.stackedWidget.setCurrentIndex(new_index)

            
            if current_index == last_page_index and direction > 0:
                self.run_job_process()

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    
    
    

    def get_file_data_frame(self, source_file_type, source_file_path, file_path, file_encoding, file_delimiter, file_has_header):
        
        try:
            
            gdf = None

            
            if source_file_type == "shp":
                gdf = load_layer_or_shp_gdf(shp_path=file_path, file_encoding=file_encoding)

            
            elif source_file_type == "layer":
                qgs_project_layer = source_file_path
                gdf = load_layer_or_shp_gdf(layer=qgs_project_layer, file_encoding=file_encoding)

            
            elif source_file_type == "json":
                df, _ = load_json_df_or_gdf(file_path=file_path, file_encoding=file_encoding)
                gdf = df_to_empty_geometry_gdf(df)

            
            elif source_file_type == "geojson":
                gdf = load_geojson_gdf(file_path=file_path, file_encoding=file_encoding)

            
            elif source_file_type == "txt":
                df = load_txt_or_csv_df(file_path, file_encoding, file_delimiter, file_has_header)
                gdf = df_to_empty_geometry_gdf(df)

            
            elif source_file_type == "csv":
                df = load_txt_or_csv_df(file_path, file_encoding, file_delimiter, file_has_header)
                gdf = df_to_empty_geometry_gdf(df)

            
            elif source_file_type == "folder":
                df = load_txt_or_csv_df(file_path, file_encoding, file_delimiter, file_has_header)
                gdf = df_to_empty_geometry_gdf(df)

            if gdf is None:
                return

            return gdf

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def run_job_process(self):
        
        try:
            
            show_progress(self.progressBar)
            update_progress(self.progressBar, self.base_progress)

            
            source_file_type, source_file_path, _ = common.fileInfo_1.file_record.get_record()
            result_file_type, result_file_path, _ = common.fileInfo_1.result_record.get_record()

            
            status_flags = []  
            for index, file_preview in enumerate(common.fileInfo_1.file_preview):

                
                file_path, file_encoding, file_delimiter, file_has_header = file_preview.get_info()

                
                if source_file_type == "folder":
                    
                    file_name_with_ext = os.path.basename(file_path)
                    new_file_path = os.path.join(result_file_path, file_name_with_ext)
                elif result_file_type == "layer":
                    new_file_path = file_path
                else:
                    new_file_path = result_file_path

                
                gdf = self.get_file_data_frame(source_file_type, source_file_path, file_path, file_encoding, file_delimiter, file_has_header)

                
                result = self.run_job_by_index(gdf, index)

                
                if result is None:
                    status_flags.append(False)
                    break
                elif result is True:
                    
                    
                    status_flags.append(True)

                try:
                    
                    if result_file_type == 'layer':

                        
                        layer_widget = self.pages_and_files[1][4].get_qgs_layer_widget()

                        
                        layer_widget_index = layer_widget.currentIndex()

                        
                        layer = source_file_path

                        
                        new_layer = update_shapefile_layer(layer, result)

                        
                        if 0 <= layer_widget_index < layer_widget.count():
                            layer_widget.setCurrentIndex(layer_widget_index)

                        
                        common.fileInfo_1.file_record.file_path[result_file_type] = new_layer

                        
                        status_flags.append(True)

                    else:
                        
                        if new_file_path:

                            
                            if isinstance(result, gpd.GeoDataFrame):
                                export_success = export_gdf(result, new_file_path)

                                
                                status_flags.append(export_success)

                            elif isinstance(result, list) and result:
                                
                                file_type, _, file_name = common.fileInfo_1.file_record.get_record()
                                base_dir = Path(new_file_path)
                                base_name = Path(file_name).stem
                                ext = f".{file_type}"

                                
                                export_success = []
                                for i, part in enumerate(result, start=1):
                                    output_path = base_dir / f"{base_name}_{i:03d}{ext}"
                                    export_success.append(export_gdf(part, output_path))

                                
                                status_flags.append(all(export_success))

                            else:
                                
                                QMessageBox.information(self, "파일 오류", "파일 저장 중 오류가 발생했습니다.", QMessageBox.Ok)
                                status_flags.append(False)

                except Exception as e:
                    
                    QMessageBox.information(self, "파일 오류", f"GeoDataFrame export 실패: {e}", QMessageBox.Ok)
                    status_flags.append(False)

                

            
            if status_flags and all(status_flags):
                update_progress(self.progressBar, 100)  
                QMessageBox.information(self, "알림", "축하합니다. 작업이 완료했습니다!", QMessageBox.Ok)

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

        finally:
            show_progress(self.progressBar, False)

    
    
    

    
    def normalize_address_field(
            self,
            gdf,
            source_field: str,
            output_field: str,
    ):
        

        
        PUNCT_RE = re.compile(r"[,\(\)\[\]\{\}]")
        SPACE_RE = re.compile(r"\s+")
        LOT_RE = re.compile(r"(산\s*)?\d+(?:-\d+)?")
        BLOCK_RE = re.compile(r"^\s*(동|층|호|단지|블록|차|관|라인|타워|빌딩)\b")

        def normalize_jibun(text: Optional[str]) -> Optional[str]:
            if not text:
                return None

            s = text.strip()
            if not s:
                return None

            s = PUNCT_RE.sub(" ", s)
            s = SPACE_RE.sub(" ", s).strip()

            matches = list(LOT_RE.finditer(s))
            if not matches:
                return text.strip()

            chosen = None
            for m in reversed(matches):
                if not BLOCK_RE.match(s[m.end():]):
                    chosen = m
                    break
            if not chosen:
                chosen = matches[-1]

            return s[:chosen.end()].strip()

        def normalize_text_cell(v):
            if v is None or v is pd.NA:
                return pd.NA
            if isinstance(v, float) and pd.isna(v):
                return pd.NA
            if isinstance(v, str):
                s = v.strip()
                if s == "":
                    return pd.NA
                return normalize_jibun(s)
            return v

        try:
            
            geometry_col = gdf.geometry.name

            
            gdf_copy = gdf.copy()

            
            gdf_attr = gdf_copy.drop(columns=geometry_col, errors='ignore')

            
            gdf_attr = normalize_null_values(gdf_attr)

            
            if output_field not in gdf_copy.columns:
                gdf_copy[output_field] = pd.NA

            
            gdf_attr[output_field] = gdf_attr[source_field].map(normalize_text_cell)

            
            gdf_copy[output_field] = gdf_attr[output_field]

            return gdf_copy

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "파일을 작업 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    def apply_address_conversion(self, gdf, input_field, mode):
        
        
        addr_list = ["" if pd.isna(v) else str(v) for v in gdf[input_field].tolist()]

        
        results = batch_geocode(addr_list, mode=mode, progressBar=self.progressBar, base_progress=self.base_progress)

        
        if results is None:
            return None

        
        if mode in ("jibun_to_road", "j2r"):
            gdf["road"] = [r.get("road") if isinstance(r, dict) else None for r in results]

        elif mode in ("road_to_jibun", "r2j"):
            gdf["jibun"] = [r.get("jibun") if isinstance(r, dict) else None for r in results]

        elif mode in ("jibun_to_coord", "j2c", "road_to_coord", "r2c"):
            gdf["lon"] = [r.get("lon") if isinstance(r, dict) else None for r in results]
            gdf["lat"] = [r.get("lat") if isinstance(r, dict) else None for r in results]

        else:
            raise ValueError("Invalid mode")

        return gdf

    
    def gdf_jibun_to_coord(self, gdf, field_name):
        
        try:
            
            geometry_col = gdf.geometry.name

            
            gdf_copy = gdf.copy()

            
            gdf_attr = gdf_copy.drop(columns=geometry_col, errors='ignore')

            
            gdf_copy_not_null_values = normalize_null_values(gdf_attr)

            
            gdf_apply = self.apply_address_conversion(
                gdf_copy_not_null_values,
                input_field=field_name,
                mode="jibun_to_coord"
            )

            
            if gdf_apply is None:
                return None

            
            gdf_copy["lon"] = gdf_apply.get("lon")
            gdf_copy["lat"] = gdf_apply.get("lat")

            return gdf_copy

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "파일을 작업 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    
    def gdf_jibun_to_road(self, gdf, source_field: str, output_field: str):
        
        try:
            
            geometry_col = gdf.geometry.name

            
            gdf_copy = gdf.copy()

            
            gdf_attr = gdf_copy.drop(columns=geometry_col, errors='ignore')

            
            gdf_attr = normalize_null_values(gdf_attr)

            
            
            gdf_apply = self.apply_address_conversion(
                gdf_attr,
                input_field=source_field,
                mode="jibun_to_road"
            )

            
            if gdf_apply is None:
                return None

            gdf_copy[output_field] = gdf_apply.get("road")

            return gdf_copy

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "파일을 작업 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    
    def gdf_road_to_coord(self, gdf, field_name):
        
        try:
            
            geometry_col = gdf.geometry.name

            
            gdf_copy = gdf.copy()

            
            gdf_attr = gdf_copy.drop(columns=geometry_col, errors='ignore')

            
            gdf_copy_not_null_values = normalize_null_values(gdf_attr)

            
            gdf_apply = self.apply_address_conversion(
                gdf_copy_not_null_values,
                input_field=field_name,
                mode="road_to_coord"
            )

            
            if gdf_apply is None:
                return None

            
            gdf_copy["lon"] = gdf_apply.get("lon")
            gdf_copy["lat"] = gdf_apply.get("lat")

            return gdf_copy

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "파일을 작업 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    
    def gdf_road_to_jibun(self, gdf, source_field: str, output_field: str):
        
        try:
            
            geometry_col = gdf.geometry.name

            
            gdf_copy = gdf.copy()

            
            gdf_attr = gdf_copy.drop(columns=geometry_col, errors='ignore')

            
            gdf_attr = normalize_null_values(gdf_attr)

            
            
            gdf_apply = self.apply_address_conversion(
                gdf_attr,
                input_field=source_field,
                mode="road_to_jibun"
            )

            
            if gdf_apply is None:
                return None

            gdf_copy[output_field] = gdf_apply.get("jibun")

            return gdf_copy

        except Exception as e:
            QMessageBox.information(self, "작업 오류", "파일을 작업 중 오류가 발생하였습니다.", QMessageBox.Ok)
            logger.error("에러 발생: %s", e, exc_info=True)
            return None

    
    
    

    @staticmethod
    def get_widget_option(job_index, job_title):
        
        try:
            option = None  
            job_title = job_title[2:]

            if job_index == 2:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": False,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": True,
                    "show_field_in_file": False,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,

                    "FILE_TUID": [
                        '지번주소 기준 필드 선택',
                        '필드 선택: ',
                        '선택한 지번주소 필드를 공간 좌표(경도·위도)로 변환하여 lon, lat 필드를 생성하고 위치 정보를 저장합니다.'
                    ],
                }
            if job_index == 3:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": False,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": True,
                    "show_field_in_file": False,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": True,
                    "output_by_table": False,

                    "FILE_TUID": [
                        '지번주소 기준 필드 선택',
                        '필드 선택: ',
                        '선택한 지번주소 필드를 도로명주소로 변환합니다.'
                    ],

                    "RESULT_FIELD": [
                        '도로명주소 결과 필드 생성',
                        '필드명 입력: ',
                        ''
                    ],
                }
            if job_index == 4:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": False,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": True,
                    "show_field_in_file": False,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": False,
                    "output_by_table": False,

                    "FILE_TUID": [
                        '도로명주소 기준 필드 선택',
                        '필드 선택: ',
                        '선택한 도로명주소 필드를 공간 좌표(경도·위도)로 변환하여 lon, lat 필드를 생성하고 위치 정보를 저장합니다.'
                    ],
                }
            if job_index == 5:
                option = {
                    "apply_basic_qss": True,

                    "disable_file_type_layer": True,
                    "disable_file_type_shp": True,
                    "disable_file_type_json": True,
                    "disable_file_type_txtcsv": True,
                    "disable_file_type_fold": False,

                    "show_uid_in_file": False,
                    "show_tuid_in_file": True,
                    "show_field_in_file": False,

                    "setting_by_text": False,
                    "setting_by_array": False,
                    "setting_by_expression": False,
                    "setting_by_section": {"enabled": False, "value_type": "int"},
                    "setting_by_numeric": {"enabled": False, "value_type": "int"},
                    "setting_by_combo": {"enabled": False, "items": []},

                    "output_by_file": True,
                    "output_by_field": True,
                    "output_by_table": False,

                    "FILE_TUID": [
                        '도로명주소 기준 필드 선택',
                        '필드 선택: ',
                        '선택한 도로명주소 필드를 지번주소로 변환합니다.'
                    ],

                    "RESULT_FIELD": [
                        '지번주소 결과 필드 생성',
                        '필드명 입력: ',
                        ''
                    ],
                }

            return option

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)

    def run_job_by_index(self, gdf, file_preview_index):
        
        try:
            
            file_info = common.fileInfo_1

            
            setting_text = file_info.file_setting.get_text()
            setting_numeric = file_info.file_setting.get_numeric()
            setting_section_min, setting_section_max = file_info.file_setting.get_section()
            setting_combo = file_info.file_setting.get_combo()
            setting_array_string, setting_array_integer, setting_array_float = file_info.file_setting.get_array()

            
            source_file_type, source_file_path, source_file_name = file_info.file_record.get_record()

            
            file_preview = file_info.file_preview[file_preview_index]
            file_field_selection = file_preview.get_selection_field()
            file_uid = file_preview.get_file_uid()
            file_tuid = file_preview.get_file_tuid()
            file_is_field_check = file_preview.get_field_check()
            result_field = file_info.result_field

            
            result = None
            if self.job_index == 2:
                gdf = keep_columns_gdf(gdf, file_field_selection) if file_is_field_check else gdf
                result = self.gdf_jibun_to_coord(gdf, file_tuid)

            elif self.job_index == 3:
                gdf = keep_columns_gdf(gdf, file_field_selection) if file_is_field_check else gdf
                result = self.gdf_jibun_to_road(gdf, file_tuid, result_field)

            elif self.job_index == 4:
                gdf = keep_columns_gdf(gdf, file_field_selection) if file_is_field_check else gdf
                result = self.gdf_road_to_coord(gdf, file_tuid)

            elif self.job_index == 5:
                gdf = keep_columns_gdf(gdf, file_field_selection) if file_is_field_check else gdf
                result = self.gdf_road_to_jibun(gdf, file_tuid, result_field)

            
            if result is None or result is False:
                return None

            return result

        except Exception as e:
            logger.error("에러 발생: %s", e, exc_info=True)
            return None




